<?php

namespace App\Http\Controllers;

use App\Http\Requests\UserRegisterRequest;
use App\Mail\RegistrationSuccessMail;
use App\Models\BankInfo;
use App\Models\Customer;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Vehicle;
use App\Utility\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class UserController extends Controller
{

    public function dashboard(){
        $user = Util::Auth();
        $data = [
            "referralCode" => $user->referral_code,
            "total_user" => $user->successfulReferrals()->count(),
            "total_income" => $user->totalReferralCommission(),
            'referree_product' => Vehicle::orderBy('id', 'desc')->where('referral_code', $user->referral_code)->with('customer', 'transaction')->limit(5)->get(),
        ];
        return view('advocate.dashboard',  $data);
    }

    public function referred()
    {
        $user = Util::Auth();
        $data = [
            'customers' => Customer::orderBy('id', 'desc')->where('referral_code', $user->referral_code)->get(),
        ];
        return view('advocate.referrers', $data);
    }

    public function referrerTransaction()
    {
        $user = Util::Auth();
        $data = [
            'transactions' => Transaction::orderBy('id', 'desc')->where('referral_code', $user->referral_code)->with('vehicle', 'customer')->get(),
        ];
        return view('advocate.transactions', $data);
    }

    public function referPolicy()
    {
        $user = Util::Auth();
        $data = [
            'vehicles' => Vehicle::orderBy('id', 'desc')->where('referral_code', $user->referral_code)->with('customer', 'transaction')->get(),
        ];
        return view('advocate.vehicles', $data);
    }

    public function userForm()
    {
        return view('advocate.index');
    }


    public function registration(UserRegisterRequest $request)
{
    $validated = $request->validated();
    DB::beginTransaction();

    try {
        $docPath = null;
        $docIdPath = null;

        if ($request->hasFile('license')) {
            $file = $request->file('naicom_license');
            $originalName = $file->getClientOriginalName();
            $docPath = $file->storeAs('licenses', $originalName, 'public');
            $validated['naicom_license'] = $docPath;
        }

        if ($request->hasFile('identification_document')) {
            $file = $request->file('identification_document');
            $originalName = $file->getClientOriginalName();
            $docIdPath = $file->storeAs('ids', $originalName, 'public');
            $validated['identification_document'] = $docIdPath;
        }

        session([
            'insurance_data' => $validated,
            'insurance_data_created_at' => now(),
        ]);

        // Save plain password separately
        $plainPassword = $request->password;

        $user = new User();
        $user->first_name = $request->first_name;
        $user->last_name = $request->last_name;
        $user->email = $request->email;
        $user->phone_number = $request->phone_number;
        $user->residential_address = $request->residential_address;
        $user->date_of_birth = $request->date_of_birth;
        $user->place_of_birth = $request->place_of_birth;
        $user->other_source_of_income = $request->other_source_of_income;
        $user->nationality = $request->nationality;
        $user->nin = $request->nin ?? null;
        $user->id_number = $request->id_number;
        $user->id_type = $request->id_type;
        $user->issued_date = $request->issued_date;
        $user->expiry_date = $request->expiry_date ?? null;
        $user->password = Hash::make($plainPassword);
        $user->license = $docPath;
        $user->identification_document = $docIdPath;
        $user->referral_code = Util::generateReferralCode();
        $user->privacy_policy = $request->privacy_policy;
        $user->channel = $request->channel ?? null;
        $user->L_number = $request->L_number ?? null;
        $user->save();

        $bank = new BankInfo();
        $bank->user_id = $user->id;
        $bank->bvn = $request->bvn ?? null;
        $bank->account_number = $request->account_number;
        $bank->bank_name = $request->bank_name;
        $bank->account_type = $request->account_type;
        $bank->save();

        // ✅ Send registration success email
        Mail::to($user->email)->send(new RegistrationSuccessMail($user, $plainPassword));

        DB::commit();

        Auth::login($user);

        return redirect()->route('advocate.dashboard')->with('success', 'Registration successful! We have sent your login details to your email.');
    } catch (\Throwable $th) {
        DB::rollBack();
        Log::error('Insurance form submission error: ' . $th->getMessage());
        return back()->withInput()->with('error', $th->getMessage());
    }
}



/**
     * Show the reset password form
     */
    public function showResetForm()
    {
        return view('reset-password');
    }

    /**
     * Handle password update
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password'      => 'required',
            'new_password'          => 'required|min:6|confirmed',
        ]);

        $user = Util::Auth();

        // Check if current password matches
        if (!Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'Your current password is incorrect']);
        }

        // Update password
        $user->password = Hash::make($request->new_password);
        $user->save();

        return redirect()->route('admin.password.reset')->with('success', 'Password updated successfully!');
    }

   
}
