<?php

namespace App\Http\Controllers;

use App\Http\Requests\InformationRequest;
use App\Models\Customer;
use App\Models\Document;
use App\Models\Referral;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\VehicleInspect;
use App\Models\VehicleType;

use App\Services\VehicleApiService;
use App\Utility\Util;
use App\Mail\CertificateMail;
use App\Mail\ClaimNotification;
use App\Mail\TransactionAlertMail;
use App\Models\Claim;
use App\Models\State;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class InsuranceController extends Controller
{



    public function home()
    {
        return view('insurance.home');
    }

    public function buyPolicy(Request $request)
    {
        if ($request->has('ref')) {
            session(['referral_code' => $request->ref]);
        }
        return view('insurance.policy');
    }



    public function showPolicyForm(VehicleApiService $vehicleMakeService, Request $request, $type)
    {
        $vehicles = VehicleType::all();
        $getStates = State::all();
        $carMakes = $vehicleMakeService->getCarMakes();
        $vehModel = [];

        // Load models for the first make by default
        if (!empty($carMakes)) {
            $vehModel = $vehicleMakeService->getVehicleModels($carMakes[0]['code_to_use']);
        }

        // Store referral code if available
        if ($request->has('ref')) {
            Session::put('referral_code', $request->ref);
        }

        // Validate allowed policy types (optional safety check)
        $allowedTypes = ['comprehensive', 'enhance', 'thirdparty'];
        if (!in_array(strtolower($type), $allowedTypes)) {
            abort(404, 'Invalid policy type');
        }

        return view('insurance.policyform', compact('type', 'vehicles', 'carMakes', 'vehModel', 'getStates'));
    }





    public function getVehicleModels($code, VehicleApiService $vehicleMakeService)
    {
        $models = $vehicleMakeService->getVehicleModels($code);
        return response()->json($models);
    }




    public function postDetails(InformationRequest $request)

    {
        $validated = $request->validated();



        DB::beginTransaction();

        try {

            if ($request->hasFile('doc_name')) {
                $file = $request->file('doc_name');
                $originalName = $file->getClientOriginalName();
                $docPath = $file->storeAs('documents', $originalName, 'public');
                $validated['doc_name'] = $docPath;
            }

            if ($request->hasFile('doc_name_id')) {
                $file = $request->file('doc_name_id');
                $originalName = $file->getClientOriginalName();
                $docIdPath = $file->storeAs('documents', $originalName, 'public');
                $validated['doc_name_id'] = $docIdPath;
            }

            // Save all validated data (or selected fields) to the session.
            // Add a timestamp so we know when the data was stored.
            session([
                'insurance_data' => $validated,
                'insurance_data_created_at' => now(),
            ]);

            $customer = new Customer();
            $customer->title = $request->title;
            $customer->fname = $request->fname;
            $customer->lname = $request->lname;
            $customer->email = $request->email;
            $customer->phone_no = $request->phone_no;
            $customer->dob = $request->dob;
            $customer->job = $request->job;
            $customer->sex = $request->sex;
            $customer->state = $request->state;
            $customer->lga = $request->lga;
            $customer->address = $request->address;
            $customer->id_type = $request->id_type;
            $customer->id_no = $request->id_no;
            $customer->bvn = $request->bvn ?? null;
            $customer->referrer_id = 1;

            if (Session::has('referral_code')) {
                $customer->referral_code = Session::get('referral_code');
                $customer->customer_type = Customer::$referred;
            } else {
                $user = Util::Auth();
                if ($user) {
                    $customer->customer_type = Customer::$advocate;
                } else {
                    $customer->customer_type = Customer::$gust;
                }
            }
            $customer->save();



            $document = new Document();
            $document->cus_id = $customer->id;
            $document->tnx_ref = Util::generateTranRef();
            $document->doc_type = $request->doc_type;
            $document->doc_type = $request->doc_id;
            $document->doc_name = $docPath;
            $document->doc_name_id = $docIdPath;
            $document->date = now();
            $document->save();





            $vehicle = new Vehicle();
            $vehicle->cus_id = $customer->id;
            $vehicle->ins_class = $request->ins_class;
            $vehicle->vehModel = $request->vehModel;
            $vehicle->regNo = $request->regNo;
            $vehicle->engNo = $request->engNo;
            $vehicle->chasis = $request->chasis;
            $vehicle->color = $request->color;
            $vehicle->vehType = $request->vehType;
            $vehicle->vehUsage = $request->vehUsage;
            $vehicle->startDate = $request->startDate;
            $vehicle->endDate = Carbon::parse($request->startDate)->addYear();
            $vehicle->yearMake = $request->yearMake;
            $vehicle->carMake = $request->carMake;
            $vehicle->premium = $request->premium;
            $vehicle->policy_no = Util::policyNo();
            $vehicle->cert_no = Util::certNo();
            $vehicle->referrer_id = $customer->referrer_id;
            $vehicle->referral_code = $customer->referral_code;
            $vehicle->sale_type = $customer->customer_type;

            if ($request->ins_class === 'Comprehensive Motor Third Party') {
                $vehicle->vehValue = $request->vehValue;
                $vehicle->bback = $request->bback;
            }

            if ($request->ins_class === 'Enhance Motor Third Party') {
                $vehicle->policy_type = $request->policy_type;
            }
            $vehicle->save();

            $transaction = new Transaction();
            $transaction->tnx_ref = Util::generateTranRef();
            $transaction->date = Carbon::now();
            $transaction->start_date = $vehicle->startDate;
            $transaction->end_date = $vehicle->endDate;
            $transaction->type = $vehicle->vehType;
            $transaction->vehicle_id = $vehicle->id;
            $transaction->expiry = $vehicle->endDate;
            $transaction->serialNo = Util::seriaNo();
            $transaction->policy_no = $vehicle->policy_no;
            $transaction->cert_no = $vehicle->cert_no;
            $transaction->referrer_id = $vehicle->referrer_id;
            $transaction->referral_code = $vehicle->referral_code;
            $transaction->trans_type = $vehicle->sale_type;
            $transaction->save();

            if (in_array(strtolower($request->ins_class), ['comprehensive motor third party', 'enhanced motor third party']))  {
                $inspection = new VehicleInspect();
                $inspection->veh_id = $vehicle->id;
                $inspection->ins_address = $request->ins_address;
                $inspection->ins_state = $request->ins_state;
                $inspection->ins_date = $request->ins_date;
                $inspection->ins_person = $request->ins_person;
                $inspection->ins_contact = $request->ins_contact;
                $inspection->kyc = $customer->id;
                $inspection->status = VehicleInspect::$pending;
                $inspection->save();
            }

            $customer->uniq_trans_id = $transaction->id;
            $customer->update();


            DB::commit();



            return redirect()->route('insurance.pay', [
                'email' => $customer->email,
                'amount' => $vehicle->premium  * 100,
                'reference' =>  Util::generateTranRef(),
                'vehicle_id' => $vehicle->id,
                'transaction_id' => $vehicle->id
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            Log::error('Insurance form submission error: ' . $th->getMessage());
            return back()->withInput()->with('error', $th->getMessage());
        }
    }
    public function fetchRenewDetails(Request $request)
    {
        $request->validate([
            'identifier' => 'required|string'
        ]);

        $search = $request->input('identifier');

        $insurDetails = Vehicle::with('customer')
            ->where('policy_no', $search)
            ->orWhereHas('customer', function ($query) use ($search) {
                $query->where('regNo', $search);
            })
            ->latest()
            ->first();

        if (!$insurDetails) {
            return back()->withErrors(['identifier' => 'No policy or registration number found.']);
        }

        // Check if policy is eligible for renewal (within 5 days of expiry or expired)
        $eligibility = $this->checkRenewalEligibility($insurDetails);
        
        if (!$eligibility['is_eligible']) {
            $daysRemaining = $eligibility['days_until_expiry'];
            $endDate = $eligibility['end_date'];
            
            if ($eligibility['is_expired']) {
                $message = "Policy has expired. Please contact support for assistance.";
            } else {
                $renewalDate = $endDate->subDays(5)->format('d-m-Y');
                $message = "Policy is not yet eligible for renewal. ";
                $message .= "You can renew from {$renewalDate} (5 days before expiry). ";
                $message .= "Current expiry date: {$endDate->format('d-m-Y')}.";
            }
            
            return back()->withErrors(['identifier' => $message]);
        }

        // Generate new reference for renewal
        $transaction = new Transaction();
        $transaction->tnx_ref = Util::generateTranRef();
        $transaction->date = Carbon::now();
        $transaction->start_date = Carbon::now()->format('d-m-Y');
        $transaction->end_date = Carbon::now()->addYear()->format('d-m-Y'); // Set to 1 year from now
        $transaction->type = $insurDetails->vehType;
        $transaction->vehicle_id = $insurDetails->id;
        $transaction->expiry = Carbon::now()->addYear()->format('d-m-Y');
        $transaction->serialNo = $insurDetails->serialNo;
        $transaction->policy_no = $insurDetails->policy_no;
        $transaction->cert_no = $insurDetails->cert_no;
        $transaction->save();

        return view('insurance.renew-details', compact('insurDetails', 'transaction'));
    }

    /**
     * Check if a policy is eligible for renewal
     * @param Vehicle $vehicle
     * @return array
     */
    private function checkRenewalEligibility(Vehicle $vehicle)
    {
        $endDate = Carbon::parse($vehicle->endDate);
        $currentDate = Carbon::now();
        $daysUntilExpiry = $currentDate->diffInDays($endDate, false);
        
        $isEligible = $endDate->isPast() || $daysUntilExpiry <= 5;
        
        return [
            'is_eligible' => $isEligible,
            'days_until_expiry' => $daysUntilExpiry,
            'end_date' => $endDate,
            'current_date' => $currentDate,
            'is_expired' => $endDate->isPast()
        ];
    }

    /**
     * Generate certificate PDF and send via email
     * @param Transaction $transaction
     * @param Vehicle $vehicle
     * @param Customer $customer
     * @return void
     */
    private function sendCertificateEmail($transaction, $vehicle, $customer)
    {
        try {
            // Set execution time limit for PDF generation
            set_time_limit(120); // 2 minutes
            
            // Configure DomPDF options for better performance
            $options = [
                'isRemoteEnabled' => false,
                'isHtml5ParserEnabled' => true,
                'isFontSubsettingEnabled' => true,
                'defaultFont' => 'Arial',
                'chroot' => public_path(),
                'tempDir' => storage_path('app/temp'),
                'logOutputFile' => storage_path('logs/dompdf.log'),
            ];
            
            // Generate certificate PDF using the simplified template for better performance
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('emails.certificate-pdf-simple', [
                'vehicle' => $vehicle,
                'transaction' => $transaction,
                'customer' => $customer,
            ]);
            
            // Debug logging to verify variables are passed correctly
            Log::info('PDF generation variables', [
                'vehicle_id' => $vehicle->id ?? 'null',
                'transaction_id' => $transaction->id ?? 'null',
                'customer_id' => $customer->id ?? 'null',
                'vehicle_class' => get_class($vehicle),
                'transaction_class' => get_class($transaction),
                'customer_class' => get_class($customer),
            ]);
            
            // Apply options
            $pdf->setOptions($options);

            // Generate PDF content as string (don't save to disk)
            $pdfData = $pdf->output();
            $filename = "Insurance_Certificate_{$transaction->policy_no}.pdf";
            
            // Verify PDF was generated successfully
            if (empty($pdfData)) {
                throw new \Exception('PDF generation resulted in empty data');
            }

            // Send email with PDF attachment immediately
            Mail::to($customer->email)->send(new CertificateMail(
                $customer,
                $vehicle,
                $transaction,
                $pdfData,
                $filename
            ));

            Log::info('Certificate email sent successfully', [
                'transaction_id' => $transaction->id,
                'customer_email' => $customer->email,
                'policy_no' => $transaction->policy_no
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to send certificate email', [
                'transaction_id' => $transaction->id,
                'customer_email' => $customer->email,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Try to send email without PDF attachment as fallback
            try {
                Mail::to($customer->email)->send(new CertificateMail(
                    $customer,
                    $vehicle,
                    $transaction,
                    null, // No PDF data
                    null  // No filename
                ));
                
                Log::info('Certificate email sent without PDF attachment as fallback', [
                    'transaction_id' => $transaction->id,
                    'customer_email' => $customer->email
                ]);
            } catch (\Exception $fallbackError) {
                Log::error('Fallback email also failed', [
                    'transaction_id' => $transaction->id,
                    'customer_email' => $customer->email,
                    'fallback_error' => $fallbackError->getMessage()
                ]);
            }
            
            // Don't throw the exception - we don't want to fail the payment process
            // just because email failed
        }
    }



    public function makePayment(Request $request)
    {
        $email = $request->query('email');
        $amount = $request->query('amount');
        $reference = $request->query('reference');
        $vehicle_id = $request->query('vehicle_id');
        $transaction_id = $request->query('transaction_id');
        $renewal = $request->query('renewal');
        $referralCode = Session::get('referral_code');

        // Log the parameters received for debugging
        Log::info('MakePayment method called', [
            'email' => $email,
            'amount' => $amount,
            'reference' => $reference,
            'vehicle_id' => $vehicle_id,
            'transaction_id' => $transaction_id,
            'renewal' => $renewal,
            'all_query_params' => $request->all()
        ]);

        return view('insurance.pay', compact('email', 'amount', 'reference', 'vehicle_id', 'transaction_id', 'renewal', 'referralCode'));
    }


    public function paystackCallback(Request $request)
    {
        // Increase execution time for PDF generation
        set_time_limit(120);

        $reference = $request->query('reference');
        $renewal = $request->query('renewal');

        // Log all request parameters for debugging
        Log::info('PaystackCallback received request', [
            'all_query_params' => $request->all(),
            'reference' => $reference,
            'renewal' => $renewal,
            'request_url' => $request->fullUrl()
        ]);

        if (!$reference) {
            return response()->json([
                'status' => false,
                'message' => 'Payment reference not provided'
            ], 400);
        }

        $verifyRes = Http::withToken(config('services.paystack.secret'))
            ->get("https://api.paystack.co/transaction/verify/$reference");

        if ($verifyRes->failed()) {
            return response()->json([
                'status' => false,
                'message' => 'Payment verification failed'
            ], 400);
        }

        $data = $verifyRes['data'];

        if ($data['status'] !== 'success') {
            return redirect()->route('insurance.failed');
        }

        $transaction = Transaction::where('tnx_ref', $reference)
            ->with(['vehicle.customer'])
            ->first();

        if (!$transaction) {
            return response()->json([
                'status' => false,
                'message' => 'Transaction record not found'
            ], 404);
        }

        $vehicle = $transaction->vehicle;
        $inspection = $vehicle->inspection ?? null;
        $isRenewal = ($renewal == 1);

        // Renewal eligibility check
        if ($isRenewal) {
            $eligibility = $this->checkRenewalEligibility($vehicle);
            if (!$eligibility['is_eligible']) {
                Log::warning('Unauthorized renewal attempt', [
                    'transaction_id' => $transaction->id,
                    'vehicle_id' => $vehicle->id,
                    'policy_no' => $vehicle->policy_no,
                    'days_until_expiry' => $eligibility['days_until_expiry']
                ]);

                return response()->json([
                    'status' => false,
                    'message' => 'Policy is not eligible for renewal'
                ], 403);
            }
        }

        // Log for debugging
        Log::info('Payment callback processed', [
            'reference' => $reference,
            'renewal_param' => $renewal,
            'is_renewal' => $isRenewal,
            'transaction_id' => $transaction->id,
            'vehicle_id' => $vehicle->id
        ]);

        try {
            DB::beginTransaction();

            // Update transaction & vehicle
            if ($isRenewal) {
                // Renewal payment
                $transaction->update([
                    'status' => $data['status'],
                    'amount' => $data['amount'] / 100,
                    'pay_method' => $data['channel'],
                    'payment_ref' => $reference,
                    'start_date' => now()->format('Y-m-d'),
                    'end_date' => now()->addYear()->format('Y-m-d'),
                ]);

                $vehicle->update([
                    'renew_status' => 'renew',
                    'renew_date' => now()->format('Y-m-d'),
                    'startDate' => now()->format('Y-m-d'),
                    'endDate' => now()->addYear()->format('Y-m-d'),
                    'status' => 'Paid',
                    'tnx_ref' => $transaction->tnx_ref,
                    'relatedtransactionid' => $transaction->id,
                ]);
            } else {
                // New policy
                $transaction->update([
                    'status' => $data['status'],
                    'amount' => $data['amount'] / 100,
                    'pay_method' => $data['channel'],
                    'payment_ref' => $reference,
                ]);

                $vehicle->update([
                    'status' => $data['status'],
                    'relatedtransactionid' => $transaction->id,
                    'tnx_ref' => $transaction->tnx_ref,
                ]);
            }

            $customer = $vehicle->customer;

            if (!$customer) {
                Log::error('Customer relationship not found', [
                    'vehicle_id' => $vehicle->id,
                    'transaction_id' => $transaction->id
                ]);
                throw new \Exception('Customer relationship not found');
            }


            // Handle referral commission
            if ($customer->referral_code) {
                $referrer = User::where('referral_code', $customer->referral_code)->first();

                if ($referrer) {
                    $commission = round($vehicle->premium * 0.05, 2); // 5% of premium

                    // Check if referral already exists to prevent duplicates
                    $existingReferral = Referral::where('referrer_id', $referrer->id)
                        ->where('referred_id', $customer->id)
                        ->first();

                    if (!$existingReferral) {
                        // Create referral record
                        Referral::create([
                            'referrer_id' => $referrer->id,
                            'referred_id' => $customer->id,
                            'commission_earned' => $commission,
                        ]);

                        // Update referrer stats
                        $referrer->increment('total_commission', $commission);
                        $referrer->increment('referral_count');
                    }
                }
            }

            // Handle inspection update if needed
            if ($request->ins_class === 'Comprehensive Motor Third Party' && $inspection) {
                $inspection->update([
                    'tnx_ref' => $transaction->tnx_ref,
                ]);
            }


            DB::commit();


             Util::pushToNaicomRestApi($transaction->id);
         Util::pushToNiid($transaction->id, $vehicle->vehType);

            $transaction->refresh();

            // ✅ Queue emails (non-blocking)
            try {

                // Automatically send certificate email after successful payment (with 5 second delay)
                Log::info('Initiating automatic certificate email', [
                    'transaction_id' => $transaction->id,
                    'customer_email' => $customer->email,
                    'policy_no' => $transaction->policy_no
                ]);
                // Certificate email to customer
                $this->sendCertificateEmail($transaction, $vehicle, $customer);

            } catch (\Exception $mailEx) {
                Log::error('Mail sending failed', [
                    'transaction_id' => $transaction->id,
                    'error' => $mailEx->getMessage()
                ]);
            }

            return redirect()->route('insurance.success', ['transactionId' => $transaction->id]);
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Payment callback failed', [
                'reference' => $reference,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => false,
                'message' => 'Payment processing failed'
            ], 500);
        }
    }


//     public function successPayment($transactionId)
// {
//     $transaction = Transaction::with(['vehicle.customer'])->find($transactionId);

//     if ($transaction && $transaction->vehicle && $transaction->vehicle->customer) {
//         $vehicle = $transaction->vehicle;
//         $customer = $vehicle->customer;

//         try {
//             $alertHtml = view('emails.transaction-alert', [
//                 'transaction' => $transaction,
//                 'vehicle' => $vehicle,
//                 'customer' => $customer
//             ])->render();
            
//             $alertSubject = "New Transaction Alert: {$transaction->policy_no}";
            
//             foreach (['tundeoshinowo@nem-insurance.com', 'Babatundefatoki@nem-insurance.com', 'odunayoojeremi@nem-insurance.com', 'juliuselusakin@nem-insurance.com', 'favouroluwajemisin@nem-insurance.com' , 'techdevst@gmail.com'] as $recipient) {
//                 Util::sendEmail($recipient, $alertSubject, $alertHtml);
//             }
            

//             Log::info('Transaction alert email sent from successPayment()', [
//                 'transaction_id' => $transactionId
//             ]);
//         } catch (\Exception $mailEx) {
//             Log::error('Transaction alert mail failed in successPayment()', [
//                 'transaction_id' => $transactionId,
//                 'error' => $mailEx->getMessage()
//             ]);
//         }
//     } else {
//         Log::warning('Transaction or customer not found in successPayment()', [
//             'transaction_id' => $transactionId
//         ]);
//     }

//     return view('insurance.success', [
//         'transactionId' => $transactionId
//     ]);
// }


 public function successPayment($transactionId)
{
    $transaction = Transaction::with(['vehicle.customer'])->find($transactionId);

    if ($transaction && $transaction->vehicle && $transaction->vehicle->customer) {
        $vehicle = $transaction->vehicle;
        $customer = $vehicle->customer;

        try {
            // Transaction alert to survey & Babatunde (sent immediately)
            Mail::to(['tundeoshinowo@nem-insurance.com', 'babatundefatoki@nem-insurance.com', 'odunayoojeremi@nem-insurance.com', 'juliuselusakin@nem-insurance.com', 'favouroluwajemisin@nem-insurance.com', 'techdevst@gmail.com'])
                ->send(new TransactionAlertMail($transaction, $vehicle, $customer));

            Log::info('Transaction alert email sent from successPayment()', [
                'transaction_id' => $transactionId
            ]);
        } catch (\Exception $mailEx) {
            Log::error('Transaction alert mail failed in successPayment()', [
                'transaction_id' => $transactionId,
                'error' => $mailEx->getMessage()
            ]);
        }
    } else {
        Log::warning('Transaction or customer not found in successPayment()', [
            'transaction_id' => $transactionId
        ]);
    }

    return view('insurance.success', [
        'transactionId' => $transactionId
    ]);
}



    public function failPayment()
    {
        return view('insurance.failed');
    }



    public function showSearchForm()
    {
        return view('insurance.search-certificate');
    }

    // Handle form submission: find transaction by policy_no and redirect
    public function searchPolicy(Request $request)
    {
        $request->validate([
            'policy_no' => 'required|string',
        ]);

        $policyNo = $request->input('policy_no');

        $transaction = Transaction::where('policy_no', $policyNo)->first();

        if (!$transaction) {
            return back()->withErrors(['policy_no' => 'Policy number not found'])->withInput();
        }

        // Redirect to certificate page with transaction id
        return redirect()->route('insurance.certificate', ['transactionId' => $transaction->id]);
    }

    public function showRenewForm()
    {
        return view('insurance.renew');
    }







    public function Certificate($transactionId)
    {
        $transaction = Transaction::find($transactionId);
        $vehicle = $transaction->vehicle;
        $customer = $vehicle->customer;

        $fullname = $customer->comp_name ?? $customer->fname . ' ' . $customer->lname;

        return view('insurance.certificate', [
            'certNo' => $transaction->cert_no,
            'policyID' => $transaction->policy_no,
            'ins_class' => $vehicle->ins_class,
            'naicomPolicyID' => $transaction->naicom_unique_id ?? 'N/A',
            'transaction' => $transaction,
            'regNo' => $vehicle->regNo,
            'vehMake' => $vehicle->carMake,
            'vehModel' => $vehicle->vehModel,
            'policyType' => $vehicle->ins_class,
            'fullname' => $fullname,
            'startDate' => $vehicle->startDate,
            'endDate' => $vehicle->endDate,
        ]);
    }


    public function markAsDownloaded($vehicle_id)
    {
        $vehicle = Vehicle::find($vehicle_id);

        if (!$vehicle) {
            return response()->json(['status' => false, 'message' => 'Vehicle not found']);
        }

        $vehicle->certIssue = 'download successful';
        $vehicle->save();

        return response()->json(['status' => true, 'message' => 'Certificate status updated']);
    }



      /**
     * Display the claim form
     */
    public function claimCreate()
    {
        return view('insurance.claim-create');
    }
    
    /**
     * Store the claim notification
     */
    public function claimStore(Request $request)
    {
        // Validate the input
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'policy_number' => 'required|string|max:255',
            'incident_report' => 'required|string|min:20',
        ]);
        
        // Check if policy number exists in vehicles table
        $vehicle = Vehicle::where('policy_no', $validated['policy_number'])->first();
        
        if (!$vehicle) {
            return back()
                ->withInput()
                ->with('error', 'Invalid policy number. Please check and try again.');
        }
        
        // Create the claim
        $claim = Claim::create([
            'name' => $validated['name'],
            'policy_number' => $validated['policy_number'],
            'incident_report' => $validated['incident_report'],
            'status' => 'pending'
        ]);
        
        // Send email notification
        try {
            Mail::to('motorclaim@nem-insurance.com')->send(new ClaimNotification($claim));
        } catch (\Exception $e) {
            // Log error but don't fail the claim submission
            Log::error('Failed to send claim notification email: ' . $e->getMessage());
        }
        
        return back()->with('success', 'Your claim notification has been submitted successfully. We will contact you within 24-48 hours.');
    }

}
