<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\Permission;
use App\Models\Role;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\VehicleInspect;
use App\Utility\Util;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class AdminController extends Controller
{
    public function index()
    {
        $data = [
            'total_user' => Customer::count(),
            'total_policy' => Vehicle::where('policy_no', '!=', null)->count(),
            'total_income' => Transaction::where('status', 'paid')->sum('amount'),
            'total_transaction' => Transaction::count(),
            'vehicle' => Vehicle::orderBy('id', 'desc')->with('customer', 'transaction')->limit(5)->get(),
            'transactions' => Transaction::where('status', 'paid')->orderBy('id', 'desc')->with('vehicle', 'customer')->limit(5)->get(),
        ];
        return view('admin.index', $data);
    }

    public function listCustomer()
    {
        $data = [
            'customers' => Customer::orderBy('id', 'desc')->get(),
        ];
        return view('admin.customers', $data);
    }

    public function referCustomer()
    {
        $data = [
           'customers' => Customer::orderBy('id', 'desc')->whereNotNull('referral_code') ->get(),

        ];
        return view('admin.customers-referred', $data);
    }

    // Your existing method
    public function listTransaction()
    {
        $data = [
            'transactions' => Transaction::orderBy('id', 'desc')->with('vehicle', 'customer')->get(),
        ];
        return view('admin.transactions', $data);
    }

    // Export to CSV
    public function exportCsv(Request $request)
    {
        $transactions = $this->getFilteredTransactions($request);
        
        $filename = 'transactions_' . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($transactions) {
            $file = fopen('php://output', 'w');
            
            // Add CSV headers
            fputcsv($file, [
                'Customer Name',
                'Title',
                'Email',
                'Phone Number',
                'Address',
                'Amount',
                'Premium Charged',
                'Serial Number',
                'Policy Type',
                'Insurance Class',
                'Transaction Reference',
                'Transaction Type',
                'Payment Method',
                'Status',
                'Type',
                'Date',
                'Start Date',
                'End Date',
                'Expiry',
                'Policy Number',
                'Certificate Number',
                'Referral Code',
            ]);

            // Add data rows
            foreach ($transactions as $transaction) {
                fputcsv($file, [
                    $transaction->vehicle->customer->fname . ' ' . $transaction->vehicle->customer->lname,
                    $transaction->vehicle->customer->title ?? '-',
                    $transaction->vehicle->customer->email ?? '-',
                    $transaction->vehicle->customer->phone_no ?? '-',
                    $transaction->vehicle->customer->address ?? '-',
                    $transaction->amount ?? '-',
                    $transaction->premiumCharged ?? '-',
                    $transaction->serialNo ?? '-',
                    $transaction->vehicle->vehType ?? '-',
                    $transaction->vehicle->ins_class ?? '-',
                    $transaction->tnx_ref ?? '-',
                    $transaction->trans_type ?? '-',
                    $transaction->pay_method ?? '-',
                    $transaction->status ?? '-',
                    $transaction->type ?? '-',
                    $transaction->date ?? '-',
                    $transaction->start_date ?? '-',
                    $transaction->end_date ?? '-',
                    $transaction->expiry ?? '-',
                    $transaction->policy_no ?? '-',
                    $transaction->cert_no ?? '-',
                    $transaction->referral_code ?? '-',
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    // Export to Excel (XLS format using HTML table)
    public function exportExcel(Request $request)
    {
        $transactions = $this->getFilteredTransactions($request);
        
        $filename = 'transactions_' . date('Y-m-d_H-i-s') . '.xls';
        
        $headers = [
            'Content-Type' => 'application/vnd.ms-excel',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $content = view('admin.transactions-excel', compact('transactions'))->render();

        return response($content, 200, $headers);
    }

    // Export to PDF (HTML to PDF conversion)
    public function exportPdf(Request $request)
    {
        $transactions = $this->getFilteredTransactions($request);
        
        return view('admin.transactions-pdf', compact('transactions'));
    }

    // Helper method to get filtered transactions
    private function getFilteredTransactions(Request $request)
    {
        $query = Transaction::orderBy('id', 'desc')->with('vehicle', 'customer');

        // Apply property filter
        if ($request->has('property') && $request->property != '') {
            $query->whereHas('vehicle', function($q) use ($request) {
                $q->where('vehType', $request->property);
            });
        }

        // Apply time filter
        if ($request->has('time') && $request->time != '') {
            switch ($request->time) {
                case 'Today':
                    $query->whereDate('date', today());
                    break;
                case 'This Week':
                    $query->whereBetween('date', [now()->startOfWeek(), now()->endOfWeek()]);
                    break;
                case 'This Month':
                    $query->whereMonth('date', now()->month)
                          ->whereYear('date', now()->year);
                    break;
            }
        }

        // Apply search filter
        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('tnx_ref', 'like', "%{$search}%")
                  ->orWhere('serialNo', 'like', "%{$search}%")
                  ->orWhereHas('vehicle.customer', function($q) use ($search) {
                      $q->where('fname', 'like', "%{$search}%")
                        ->orWhere('lname', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        return $query->get();
    }

    public function listVehicle()
    {
        $data = [
            'vehicles' => Vehicle::orderBy('id', 'desc')->with('customer', 'transaction')->get(),
        ];
        return view('admin.vehicles', $data);
    }

    public function inspection()
    {
        $data = [
            'inspect' => VehicleInspect::orderBy('id', 'desc')->with('vehicle')->get(),
        ];
        return view('admin.inspect', $data);
    }

    public function advocate()
    {
        $data = [
            'advocate' => User::orderBy('id', 'desc')->where('user_role', User::$User_Role)->get(),
        ];
        return view('admin.advocate', $data);
    }




    public function updateCustomer(Request $request)
    {
        // Validate that the customer ID exists
        $request->validate([
            'id' => 'required|exists:customers,id',
        ]);

        // Find the customer
        $customer = Customer::find($request->id);

        // Conditionally update fields only if present
        $customer->title    = $request->title    ?? $customer->title;
        $customer->fname    = $request->fname    ?? $customer->fname;
        $customer->lname    = $request->lname    ?? $customer->lname;
        $customer->email    = $request->email    ?? $customer->email;
        $customer->phone_no = $request->phone_no ?? $customer->phone_no;
        $customer->dob      = $request->dob      ?? $customer->dob;
        $customer->address  = $request->address  ?? $customer->address;
        $customer->job      = $request->job      ?? $customer->job;
        $customer->sex   = $request->sex   ?? $customer->sex;

        // Save the updates
        $customer->save();

        return redirect()->back()->with('success', 'Customer Information updated successfully!');
    }

    public function inspectUpdate(Request $request)
    {
        $user = Util::Auth();
        $request->validate([
            'physical_check' => 'required|string|max:255',
            'proof' => 'string',
            // 'proof' => 'nullable|mimes:jpg,jpeg,png,pdf',
            'report' => 'nullable|string',
            'status' => 'nullable|string',
        ]);

        $inspection = VehicleInspect::find($request->id);

        try {
        $inspection->admin_id = $user->id;
        if ($request->hasFile('proof')) {
            $proofPath = $request->file('proof')->store('proofs', 'public');
            $inspection->proof = $proofPath ?? $inspection->proof;
        }

        $inspection->physical_check = $request->physical_check ?? $inspection->physical_check;
        $inspection->report = $request->report ?? $inspection->report ;
        $inspection->status = $request->status ?? $inspection->status ;
        $inspection->save();

        return redirect()->back()->with('success', 'Vehicle inspection updated successfully.');
    } catch (\Throwable $th) {
        Log::error('Insurance form submission error: ' . $th->getMessage());
        return back()->withInput()->with('error', $th->getMessage());
    }
    }


    public function response()
{
    $vehicles = Vehicle::with('customer') // make sure we eager load the relationship
        ->orderBy('id', 'desc')
        ->get();

    return view('admin.response', compact('vehicles'));
}








public function repush($id)
{
    try {
        // ✅ Find the vehicle first
        $vehicle = Vehicle::with('transaction')->find($id);

        if (!$vehicle) {
            Log::warning("Repush failed - vehicle not found", ['vehicle_id' => $id]);
            return back()->with('error', 'Vehicle not found.');
        }

        $messages = [];

        // ✅ Get the transaction id safely
        $transactionId = $vehicle->transaction->id ?? $vehicle->relatedtransactionid ?? null;
        if (!$transactionId) {
            Log::warning("Repush failed - vehicle has no linked transaction", [
                'vehicle_id' => $vehicle->id,
                'policy_no'  => $vehicle->policy_no
            ]);
            return back()->with('error', 'Vehicle has no linked transaction.');
        }

        // ✅ Log what we are about to do
        Log::info("Repush starting for vehicle", [
            'vehicle_id'     => $vehicle->id,
            'policy_no'      => $vehicle->policy_no,
            'transaction_id' => $transactionId,
            'naicom_id'      => $vehicle->naicom_unique_id,
            'niid_resp'      => $vehicle->niid_resp
        ]);

        // ✅ Push to NIID if missing
        if (empty($vehicle->niid_resp)) {
            try {
                Util::pushToNiid($vehicle->id, $vehicle->vehType);
                Log::info("NIID repush triggered for vehicle policy no {$vehicle->policy_no}");
                $messages[] = "NIID repush triggered";
            } catch (\Throwable $e) {
                Log::error("NIID repush failed for policy no {$vehicle->policy_no}", [
                    'error' => $e->getMessage()
                ]);
            }
        }

        // ✅ Push to NAICOM if missing
        if (empty($vehicle->naicom_unique_id)) {
            try {
                $response = Util::pushToNaicomRestApi($transactionId);
                Log::info("NAICOM repush triggered for vehicle policy no {$vehicle->policy_no}", [
                    'transaction_id' => $transactionId,
                    'response'       => $response instanceof \Illuminate\Http\JsonResponse
                        ? $response->getData(true)
                        : $response
                ]);
                $messages[] = "NAICOM repush triggered";
            } catch (\Throwable $e) {
                Log::error("NAICOM repush failed for policy no {$vehicle->policy_no}", [
                    'error' => $e->getMessage()
                ]);
            }
        }

        if (empty($messages)) {
            return back()->with('info', 'Nothing to repush – both NIID and NAICOM already have responses.');
        }

        return back()->with('success', implode(' & ', $messages) . " successfully!");
    } catch (\Throwable $th) {
        Log::error("Repush controller crashed", [
            'vehicle_id' => $id,
            'error' => $th->getMessage(),
            'trace' => $th->getTraceAsString()
        ]);
        return back()->with('error', 'Failed to repush. Check logs for details.');
    }
}









// Add new admin user

public function listAdmins()
{
    $admins = User::where('user_role', 'admin')->get();
    return view('admin.admins', compact('admins'));
}

public function store(Request $request)
    {
        $request->validate([
            'first_name'     => 'required|string|max:255',
            'email'    => 'required|email|unique:users,email',
            'password' => 'required|min:6',
            'role'     => 'required|string',
        ]);

        // 1. Find or create the role
        $role = Role::firstOrCreate(['name' => $request->role]);

        // 2. Create the user
        $user = new User();
        $user->first_name = $request->first_name;
        $user->email = $request->email;
        $user->password = Hash::make($request->password);
        $user->role_id = $role->id;
        $user->user_role = User::$Admin_Role;
        $user->save();

        // 3. Handle Permissions
        if ($request->has('permissions')) {
            foreach ($request->permissions as $page => $actions) {
                foreach ($actions as $action) {
                    $permission = Permission::firstOrCreate([
                        'page' => $page,
                        'action' => $action,
                    ]);
                    $role->permissions()->syncWithoutDetaching([$permission->id]);
                }
            }
        }

        return redirect()->back()->with('success', 'Admin user added successfully!');
    }



    public function update(Request $request)
{
    $request->validate([
        'id'    => 'required|exists:users,id',
        'first_name'  => 'required|string|max:255',
        'email' => 'required|email|unique:users,email,' . $request->id,
        'role'  => 'required|string',
    ]);

    // 1. Find the user
    $user = User::find($request->id);

    // 2. Update basic info
    $user->first_name = $request->first_name;
    $user->email = $request->email;

    // Optionally update password if provided
    if ($request->filled('password')) {
        $request->validate(['password' => 'min:6']);
        $user->password = Hash::make($request->password);
    }

    // 3. Update role
    $role = Role::firstOrCreate(['name' => $request->role]);
    $user->role_id = $role->id;
    $user->user_role = User::$Admin_Role;
    $user->save();

    // 4. Handle Permissions
    // Remove old permissions from role first
    $role->permissions()->detach();

    if ($request->has('permissions')) {
        foreach ($request->permissions as $page => $actions) {
            foreach ($actions as $action) {
                $permission = Permission::firstOrCreate([
                    'page' => $page,
                    'action' => $action,
                ]);
                $role->permissions()->syncWithoutDetaching([$permission->id]);
            }
        }
    }

    return redirect()->back()->with('success', 'Admin user updated successfully!');
}


}
